  ' -----------------------------------------------------------
  ' E32TTL100 LoRa module driving program.bas 
  ' A program for Micromite to communicate with the E32-TTL-100
  ' 434MHz LoRa data transceiver module, via the COM2 serial
  ' interface, for checking its capabilities.
  ' Written by Jim Rowe for Silicon Chip.
  ' (With help from Geoff Graham - many thanks, Geoff)
  ' Last revision 14/12/2018 at 8:35 am
  '
  ' NOTES:
  ' 1. This program uses the following connections between the
  ' E32-TTL-100 module and the Micromite:
  ' E32-TTL-100 pin       Micromite pin
  ' M0 (Mode 0 pin)       Pin 18
  ' M1 (Mode 1 pin)       Pin 17
  ' RXD (serial receive)  Pin 9 (COM2 TX)
  ' TXD (serial transmit) Pin 10 (COM2 RX)
  ' AUX (AUX pin)         Pin 16
  ' VCC (+3.3V)           +5V pins (via an LM1117T-3.3 regulator)
  ' GND                   GND pins
  '
  ' 2. For legal use in Australia, the E22-TTL-100 module should be
  ' programmed to operate at a centre frequency of 434MHz and with an
  ' RF output power of +14dBm or 25mW. This program does so, using the
  ' following command string:
  ' &HC0 - Set parameters command byte (header)
  ' &H00 - Set module address bytes
  ' &H00
  ' &H1C - SPED byte (8N1, 9600 baud UART, 9600 baud air rate)
  ' &H18 - CHAN byte (freq = 410 + 24 (= 18h) = 434MHz)
  ' &H46 - OPTION byte (transp TX, TXD & AUX pushpull outputs, wireless wakeup
  '       time 250ms, FEC on, TX power = 14dBm or 25mW)
  '
  ' 3: The program uses a software serial port (#2) to communicate with 
  ' the E32-TTL-100 module, at 9600 baud.
  '
  ' 4: Before the E32-TTL-100 module is used in 'normal' (transparent coms)
  ' mode (Mode 0), it should have an antenna connected to the SMA RF output
  ' socket. Otherwise the module may be damaged.
  ' 
  ' ---------------------------------------------------------------------------  
  
  OPTION AUTORUN ON
  OPTION EXPLICIT
  
  DIM AS INTEGER index = 0          ' counter
  DIM AS INTEGER NxtByte = 0        ' for bytes sent or returned
  
  DIM AS INTEGER SetParam(6)
  SetParam(0) = &HC0
  SetParam(1) = &H00
  SetParam(2) = &H00
  SetParam(3) = &H1C
  SetParam(4) = &H18
  SetParam(5) = &H46
  DIM AS INTEGER ReqParam = &HC1
  DIM AS INTEGER ReqFWVer = &HC3
  
  DIM AS STRING TxStrng$             ' text string to transmit in Mode 0
  DIM AS STRING RxStrng$             ' receive text string in Mode 0 
  DIM AS STRING junk$                ' dummy string used to clear buffers
  
  Const DBlue = RGB(0,0,128)
  CONST Bone = RGB(255,255,192)
  CONST White = RGB(WHITE)
  CONST Black = RGB(BLACK)
  CONST Red = RGB(RED)
  CONST Green = RGB(GREEN)
  
  SETPIN 16, DIN                ' declare pin 16 an input (for AUX)
  PIN(17) = 0: SETPIN 17, DOUT  ' set pin 17 low & declare it an output (M1)
  PIN(18) = 0: SETPIN 18, DOUT  ' set pin 18 low & declare it an output (M0)

  OPEN "COM2:9600" AS #2    ' open COM2 serial port, at 9600 baud
  
  ' now initialise the E32-TTL-100 control parameters
  PIN(17) = 1   'set M0 & M1 pins to 1 (high), for module's Mode 3
  PIN(18) = 1
  Wait4Aux      ' wait for AUX pin to go high, showing module ready to go
  PRINT "Module in Mode 3 for writing control parameters"
  PRINT " "
  
  PRINT "Set parameters command packet sent: ";
  FOR index = 0 TO 5 STEP 1 ' now send the control & parameter bytes
    NxtByte = SetParam(index)
    PRINT #2, CHR$(NxtByte);
    PRINT HEX$(NxtByte,2); " ";   
  NEXT
  PRINT " "
  
  PRINT "Request saved parameters cmd packet sent: ";
  FOR index = 0 TO 2 STEP 1   ' now send request for saved parameters
    NxtByte = ReqParam
    PRINT #2, CHR$(NxtByte);
    PRINT HEX$(NxtByte,2); " ";
  NEXT
  PRINT " "
  
  PRINT "Saved parameters are: "; 
  FOR index = 0 TO 5 STEP 1
    NxtByte = ASC(INPUT$(1, #2))   ' then fetch parameter byte
    PRINT HEX$(NxtByte,2); " ";    ' & send to console
  NEXT
  PRINT " "
  junk$ = INPUT$(255, #2)   ' clear the Rx buffer
  
  PRINT "Request module's firmware version packet sent: ";
  FOR index = 0 TO 2 STEP 1   ' now request module's firmware version info
  NxtByte = ReqFWVer
  PRINT #2, CHR$(NxtByte);
  PRINT HEX$(NxtByte,2); " ";
  NEXT
  PRINT " "
  junk$ = INPUT$(255, #2)   ' clear the Rx buffer again
  
  PRINT "Firmware version packet returned = ";
  FOR index = 0 TO 3  
    NxtByte = ASC(INPUT$(1, #2))   ' fetch FW version byte
    PRINT HEX$(NxtByte,2); " ";    ' & send to console
  NEXT
  PRINT " "
  PRINT " "

  PIN(17) = 0             'set M0 & M1 pins to 0 (low), for Mode 0
  PIN(18) = 0
  Wait4Aux      ' wait for AUX pin to go high again
  PRINT "Switching to Mode 0, for transparent LoRa Tx and Rx"
  PRINT " "
  junk$ = INPUT$(255, #2)   ' clear the Rx buffer again  
  ' -----------------------------------------------------------------
  ' main program loop starts here
  ShowOpngScrn          ' first show the opening screen
  DO
    PRINT INPUT$(1, #2);    
    PRINT #2, INKEY$;
  LOOP
END ' end of main part of program; subroutines and functions follow
  
  ' *****************************************************************
  ' subroutine to show the opening LCD screen with buttons
SUB ShowOpngScrn 
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), DBlue
  TEXT MM.HRes/2, MM.VRes/8, "SILICON CHIP", CM, 1, 3, Red, DBlue
  TEXT MM.HRes/2, MM.VRes/4, "E32-TTL-100", CM, 1, 2, White, DBlue
  TEXT MM.HRes/2, MM.VRes*3/8, "LoRa Module", CM, 1, 2, White, DBlue
  TEXT MM.HRes/2, MM.VRes*5/8, "Checkout Program", CM, 1, 2, White, DBlue  

END SUB
  ' -----------------------------------------------------------------
  ' subroutine to wait until AUX line goes high, to indicate module free
SUB Wait4Aux
  DO      ' wait for AUX to go high
  LOOP UNTIL PIN(16) = 1
END SUB
  ' -----------------------------------------------------------------
